﻿/*  Author:     Ben Hall
 *  File:       MainPage.xaml.cs
 *  Date:       4/28/2011
 *  
 *  Purpose:    The MainPage.xaml.cs is used by the Silverlight program to initialize the program
 *              when it begins, as well as handle user input such as button clicks.  There are many 
 *              methods in this class to handle each user event.  It also contains methods used by
 *              the event handlers, such as methods for creating new characters, or methods used to
 *              move or check for map events.
 *              
 *              MainPage.xaml.cs is also a partial class, with the other parts being MainPage.Combat.cs
 *              and MainPage.Chat.cs.
 */

using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Animation;
using System.Windows.Shapes;
using System.Windows.Media.Imaging;
using System.Windows.Resources;
using System.IO;
using System.IO.IsolatedStorage;

namespace OnlineGame
{
    public partial class MainPage : UserControl
    {
        public static PlayerCharacter playerCharacter;
        public static GameMap currentMap;
        public static Position currentPos;
        private static Boolean mapMode = true;
        public static Random rnd;

        //These arrays are initialized when the program starts, and hold various game data
        public static Skill[] skillList;
        public static Item[] itemList;
        public static AICharacter[] opponentList;
        public static GameMap[] mapList;
        public static Equipment[] equipmentList;


        public MainPage()
        {
            //initialize Random number generator
            rnd = new Random();
            InitializeComponent();

            //initialize a blank playerCharacter, which will be replaced with a new or loaded character
            playerCharacter = new PlayerCharacter();

            //import list of skills
            LoadSkillList();

            //import list of items
            LoadItemList();

            //import list of Equipment
            LoadEquipmentList();

            //import list of AICharacters
            LoadAICharacterList();

            //import list of Maps
            LoadMapList();

            //check to see if saved character file exists in IsolatedStorage.  If not, disable LoadCharacter button
            try
            {
                IsolatedStorageFile storageFile = IsolatedStorageFile.GetUserStoreForApplication();
                if (storageFile.FileExists("SavedCharacter.txt") == true)
                {
                    LoadCharacterButton.IsEnabled = true;
                }
                else
                {
                    LoadCharacterButton.IsEnabled = false;
                    MainPageErrorBox.Text = "No saved character data exists.";
                }
            }
            catch
            {
                //if the user does not allow Silverlight programs to store data, give a warning on the welcome page
                LoadCharacterButton.IsEnabled = false;
                MainPageErrorBox.Text = "Please enable the application storage feature of Silverlight to save your character data";
            }
            

            NewCharacterGrid.Visibility = Visibility.Collapsed;
            //CampButton is initially not enabled, it should enable after a successful search or after combat.
            CampButton.IsEnabled = false;

            currentMap = new GameMap();
            currentPos = new Position(7, 5);
            mapMode = true;
    
        }


        /*  GetCurrentCharacterSaveInfo() returns a string holding the information needed to reconstruct
         *  the PlayerCharacter based on the current character's information.
         */
        private String GetCurrentCharacterSaveInfo()
        {
            String info = "";

            //name
            info += playerCharacter.GetName() + "\n";

            //class, level
            if (playerCharacter.GetPlayerClass() == PlayerClass.Fighter)
            {
                info += "Fighter";
            }
            else if (playerCharacter.GetPlayerClass() == PlayerClass.Rogue)
            {
                info += "Rogue";
            }
            else if (playerCharacter.GetPlayerClass() == PlayerClass.Cleric)
            {
                info += "Cleric";
            }
            else if (playerCharacter.GetPlayerClass() == PlayerClass.Wizard)
            {
                info += "Wizard";
            }
            info += "," + playerCharacter.GetLevel() + "\n";
            
            //experience, gold
            info += playerCharacter.GetCurrentExperience() + "," + playerCharacter.GetCurrentCurrencyAmount() + "\n";
            
            //strength, stamina, agility, intelligence, faith, persona
            info += playerCharacter.GetStrength() + "," + playerCharacter.GetStamina() + "," + playerCharacter.GetAgility() + "," + playerCharacter.GetIntelligence() + "," + playerCharacter.GetFaith() + "," + playerCharacter.GetPersona() + "\n";
            
            //portrait name, icon name
            //add portrait first
            String imageString = playerCharacter.GetPortrait().UriSource.OriginalString;
            //split String from Images/imagename.jpg to get only the name.
            String[] imageName = imageString.Split('/');
            info += imageName[imageName.Length - 1];
            //add character icon
            imageString = playerCharacter.GetIcon().UriSource.OriginalString;
            imageName = imageString.Split('/');
            info += "," + imageName[imageName.Length - 1] + "\n";

            //skills known
            String stringToAdd = "";
            bool firstSkillKnown = false;
            for (int i = 0; i < playerCharacter.GetSkillsKnown().Length; i++)
            {
                if (playerCharacter.IsSkillKnown(i) == true)
                {
                    if (firstSkillKnown == true)
                    {
                        stringToAdd += ",";
                    }
                    stringToAdd += skillList[i].GetSkillName();
                    firstSkillKnown = true;
                }
            }
            //if the string is empty, use NULL in place, to signify no skills are known
            if (stringToAdd.Equals(""))
            {
                stringToAdd = "NULL";
            }
            info += stringToAdd + "\n";
            //current map ID, current position values
            info += playerCharacter.GetCurrentMap().GetMapID() + ",";
            //info += playerCharacter.GetCurrentPosition().X + "," + playerCharacter.GetCurrentPosition().Y + "\n";
            info += currentPos.X + "," + currentPos.Y + "\n";
            
            //current weapon or NULL
            if (playerCharacter.GetEquippedWeapon() == null)
            {
                info += "NULL";
            }
            else
            {
                info += playerCharacter.GetEquippedWeapon().GetItemName();
            }
            info += "\n";

            //current body armor or NULL
            if (playerCharacter.GetEquippedBodyArmor() == null)
            {
                info += "NULL";
            }
            else
            {
                info += playerCharacter.GetEquippedBodyArmor().GetItemName();
            }
            info += "\n";

            //current shield or NULL
            if (playerCharacter.GetEquippedShield() == null)
            {
                info += "NULL";
            }
            else
            {
                info += playerCharacter.GetEquippedShield().GetItemName();
            }
            info += "\n";
            
            //item inventory (Ex: 0:5, 1:2, 4:3)
            stringToAdd = "";
            bool firstItemAdded = false;
            for (int i = 0; i < playerCharacter.GetItemInventory().Length; i++)
            {
                if (playerCharacter.HasItem(i) == true)
                {
                    if (firstItemAdded == true)
                    {
                        stringToAdd += ",";
                    }
                    stringToAdd += i + ":" + playerCharacter.GetItemQuantity(i);
                    firstItemAdded = true;
                }
            }
            if (stringToAdd.Equals(""))
            {
                stringToAdd = "NULL";
            }
            info += stringToAdd + "\n";

            //equipment inventory
            stringToAdd = "";
            bool firstEquipmentAdded = false;
            for (int i = 0; i < playerCharacter.GetEquipmentInventory().Length; i++)
            {
                if (playerCharacter.HasEquipment(i) == true)
                {
                    if (firstEquipmentAdded == true)
                    {
                        stringToAdd += ",";
                    }
                    stringToAdd += i + ":" + playerCharacter.GetEquipmentQuantity(i);
                    firstEquipmentAdded = true;
                }
            }
            if (stringToAdd.Equals(""))
            {
                stringToAdd = "NULL";
            }
            info += stringToAdd + "\n";

            return info;
        }

        /*  SavePlayerCharacter() saves the String returned by GetCurrentCharacterSaveInfo() into the file
         *  "SavedCharacter.txt" in IsolatedStorage, which can then be loaded the next time the user loads
         *  the program.
         */
        private void SavePlayerCharacter()
        {
            //using IsolatedStorage on user side
            try
            {
                IsolatedStorageFile storageFile = IsolatedStorageFile.GetUserStoreForApplication();

                using (IsolatedStorageFileStream fs = storageFile.OpenFile("SavedCharacter.txt", FileMode.OpenOrCreate))
                {
                    StreamWriter sw = new StreamWriter(fs);
                    sw.Write(GetCurrentCharacterSaveInfo());
                    sw.Close();
                }
            }
            catch
            {
                //AddMessageToChatBox("Character save error", systemMessageColor);
            }

        }

        /*  LoadPlayerCharacter() loads the GameCharacter representing the player's character, including
         *  adding a full list of skills known, inventory and so on.
         */
        private Boolean LoadPlayerCharacter()
        {
            try
            {
                IsolatedStorageFile storageFile = IsolatedStorageFile.GetUserStoreForApplication();

                //if the file SavedCharacter.txt does not exist, do not attempt to load it
                if (storageFile.FileExists("SavedCharacter.txt"))
                {
                    using (IsolatedStorageFileStream fs = storageFile.OpenFile("SavedCharacter.txt", FileMode.Open))
                    {
                        StreamReader sr = new StreamReader(fs);
                        //Get info to recreate character
                        String currentLine = sr.ReadLine();
                        String[] currentLineParts;
                        //name
                        String loadedName = currentLine;
                        
                        //class and level
                        currentLine = sr.ReadLine();
                        currentLineParts = currentLine.Split(',');
                        PlayerClass loadedClass = PlayerClass.Fighter;
                        if (currentLineParts[0].Equals("Fighter"))
                        {
                            loadedClass = PlayerClass.Fighter;
                        }
                        else if (currentLineParts[0].Equals("Rogue"))
                        {
                            loadedClass = PlayerClass.Rogue;
                        }
                        else if (currentLineParts[0].Equals("Cleric"))
                        {
                            loadedClass = PlayerClass.Cleric;
                        }
                        else if (currentLineParts[0].Equals("Wizard"))
                        {
                            loadedClass = PlayerClass.Wizard;
                        }
                        int loadedLevel = Convert.ToInt32(currentLineParts[1]);

                        //current experience, current gold
                        currentLine = sr.ReadLine();
                        currentLineParts = currentLine.Split(',');
                        int loadedExperience = Convert.ToInt32(currentLineParts[0]);
                        int loadedGold = Convert.ToInt32(currentLineParts[1]);

                        //strength, stamina, agility, intelligence, faith, persona
                        currentLine = sr.ReadLine();
                        currentLineParts = currentLine.Split(',');
                        int loadedStr = Convert.ToInt32(currentLineParts[0]);
                        int loadedStm = Convert.ToInt32(currentLineParts[1]);
                        int loadedAgl = Convert.ToInt32(currentLineParts[2]);
                        int loadedInt = Convert.ToInt32(currentLineParts[3]);
                        int loadedFth = Convert.ToInt32(currentLineParts[4]);
                        int loadedPrs = Convert.ToInt32(currentLineParts[5]);

                        //create PlayerCharacter
                        PlayerCharacter loadedCharacter = new PlayerCharacter(loadedName, loadedClass, loadedLevel, loadedStr, loadedStm, loadedAgl, loadedInt, loadedFth, loadedPrs, loadedExperience, loadedGold);

                        //portrait name, icon name
                        currentLine = sr.ReadLine();
                        currentLineParts = currentLine.Split(',');
                        String portraitName = currentLineParts[0];
                        String iconName = currentLineParts[1];

                        BitmapImage loadedPortrait = new BitmapImage(new Uri("Images/" + portraitName, UriKind.RelativeOrAbsolute));
                        BitmapImage loadedIcon = new BitmapImage(new Uri("Images/" + iconName, UriKind.RelativeOrAbsolute));

                        loadedCharacter.SetPortrait(loadedPortrait);
                        loadedCharacter.SetIcon(loadedIcon);

                        //Skills known (if none are known, returns NULL)
                        currentLine = sr.ReadLine();
                        
                        //skip adding skills if null
                        if (!currentLine.Equals("NULL"))
                        {
                            currentLineParts = currentLine.Split(',');
                            for (int i = 0; i < currentLineParts.Length; i++)
                            {
                                loadedCharacter.AddSkill(currentLineParts[i]);
                            }
                        }

                        //current map, current position
                        currentLine = sr.ReadLine();
                        currentLineParts = currentLine.Split(',');
                        
                        GameMap loadedMap = GetMapWithID(currentLineParts[0]);
                        Position loadedPosition = new Position(Convert.ToInt32(currentLineParts[1]), Convert.ToInt32(currentLineParts[2]));
                        loadedCharacter.SetCurrentMap(loadedMap);
                        loadedCharacter.SetCurrentPosition(loadedPosition);

                        //equipped weapon
                        currentLine = sr.ReadLine();
                        if (!currentLine.Equals("NULL"))
                        {
                            Weapon loadedWeapon = (Weapon)equipmentList[loadedCharacter.GetEquipmentWithName(currentLine)];
                            loadedCharacter.Equip(loadedWeapon);
                        }

                        //equipped body armor
                        currentLine = sr.ReadLine();
                        if (!currentLine.Equals("NULL"))
                        {
                            Armor loadedBodyArmor = (Armor)equipmentList[loadedCharacter.GetEquipmentWithName(currentLine)];
                            loadedCharacter.Equip(loadedBodyArmor);
                        }

                        //equipped shield
                        currentLine = sr.ReadLine();
                        if (!currentLine.Equals("NULL"))
                        {
                            Armor loadedShield = (Armor)equipmentList[loadedCharacter.GetEquipmentWithName(currentLine)];
                            loadedCharacter.Equip(loadedShield);
                        }

                        //item inventory
                        currentLine = sr.ReadLine();
                        if (!currentLine.Equals("NULL"))
                        {
                            currentLineParts = currentLine.Split(',');
                            String[] itemLineParts;
                            for (int i = 0; i < currentLineParts.Length; i++)
                            {
                                itemLineParts = currentLineParts[i].Split(':');
                                loadedCharacter.AddItem(Convert.ToInt32(itemLineParts[0]), Convert.ToInt32(itemLineParts[1]));
                            }
                        }

                        //equipment inventory
                        currentLine = sr.ReadLine();

                        if (!currentLine.Equals("NULL"))
                        {
                            currentLineParts = currentLine.Split(',');
                            String[] equipmentLineParts;
                            for (int i = 0; i < currentLineParts.Length; i++)
                            {
                                equipmentLineParts = currentLineParts[i].Split(':');
                                loadedCharacter.AddEquipment(Convert.ToInt32(equipmentLineParts[0]), Convert.ToInt32(equipmentLineParts[1]));
                            }
                        }

                        //set current playerCharacter to the loadedCharacter
                        playerCharacter = loadedCharacter;

                    }
                }
                else
                {
                    return false;
                }
            }
            catch   
            {
                MainPageErrorBox.Text = "Error in loading character.";
                return false;
            }

            return true;
        }

        /*  GetCopyOfAIFromList() returns a copy of the AICharacter in opponentList at the given index.
         *  This is used for creating additional instances of AICharacters to be passed into combat
         *  encounters.
         */
        public AICharacter GetCopyOfAIFromList(int indexInList)
        {
            AICharacter AIToCopy = opponentList[indexInList];
            AICharacter AIToReturn;
            if (AIToCopy is AttackerAI)
            {
                AIToReturn = new AttackerAI(AIToCopy.GetName(), AIToCopy.GetStrength(), AIToCopy.GetStamina(), AIToCopy.GetAgility(), AIToCopy.GetIntelligence(), AIToCopy.GetFaith(), AIToCopy.GetPersona(), AIToCopy.GetCurrencyCarried());
            }
            else if (AIToCopy is BalancedAI)
            {
                AIToReturn = new BalancedAI(AIToCopy.GetName(), AIToCopy.GetStrength(), AIToCopy.GetStamina(), AIToCopy.GetAgility(), AIToCopy.GetIntelligence(), AIToCopy.GetFaith(), AIToCopy.GetPersona(), AIToCopy.GetCurrencyCarried());
            }
            else if (AIToCopy is DefenderAI)
            {
                AIToReturn = new DefenderAI(AIToCopy.GetName(), AIToCopy.GetStrength(), AIToCopy.GetStamina(), AIToCopy.GetAgility(), AIToCopy.GetIntelligence(), AIToCopy.GetFaith(), AIToCopy.GetPersona(), AIToCopy.GetCurrencyCarried());
            }
            else if (AIToCopy is HealerAI)
            {
                AIToReturn = new HealerAI(AIToCopy.GetName(), AIToCopy.GetStrength(), AIToCopy.GetStamina(), AIToCopy.GetAgility(), AIToCopy.GetIntelligence(), AIToCopy.GetFaith(), AIToCopy.GetPersona(), AIToCopy.GetCurrencyCarried());
            }
            else
            {
                AIToReturn = null;
            }
            if (AIToReturn != null)
            {
                AIToReturn.SetPortrait(AIToCopy.GetPortrait());
                AIToReturn.SetDamage(AIToCopy.GetMinDamage(), AIToCopy.GetMaxDamage());
                AIToReturn.SetArmor(AIToCopy.GetArmor());
                //add skills
                bool[] AIToCopySkillsKnown = AIToCopy.GetSkillsKnown();
                for (int i = 0; i < AIToCopySkillsKnown.Length; i++)
                {
                    if (AIToCopySkillsKnown[i] == true)
                    {
                        AIToReturn.AddSkill(i);
                    }
                }
                AIToReturn.DetermineLevel();
            }

            return AIToReturn;
        }

        /*  GetCopyOfAIFromList() returns a copy of the first AICharacter in opponentList with the same name
         *  as the given string.  This is used for creating additional instances of AICharacters to be
         *  passed into combat encounters.
         */
        public AICharacter GetCopyOfAIFromList(String AINameIn)
        {
            int AIIndex = 0;
            for (int i = 0; i < opponentList.Length; i++)
            {
                if ((opponentList[i] != null) && (opponentList[i].GetName().Equals(AINameIn)))
                {
                    AIIndex = i;
                    break;
                }
            }
            return GetCopyOfAIFromList(AIIndex);
        }

        /*  CountLinesInFile() takes in a Uri file object and returns the number of lines contained in the
         *  file.
         */
        private int CountLinesInFile(Uri fileIn)
        {
            StreamReader sr = new StreamReader(App.GetResourceStream(fileIn).Stream);
            int count = 0;
            String line = sr.ReadLine();
            while (line != null)
            {
                count++;
                line = sr.ReadLine();
            }
            return count;
        }

        /*  LoadMapList() loads a list of all GameMap objects needed in the game from a .txt resource file
         *  Maps.txt located in /GameData/Maps
         *  GameMaps are loaded into the mapList array which is initialized to the size equal to the number
         *  of lines in the Maps.txt file
         */
        private void LoadMapList()
        {
            try
            {
                Uri uriTextFile = new Uri("GameData/Maps/Maps.txt", UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);

                String mapLineToLoad = sr.ReadLine();
                int mapListArrayPosition = 0;
                mapList = new GameMap[CountLinesInFile(uriTextFile)];
                while (mapLineToLoad != null)
                {
                    String[] mapLineParts = mapLineToLoad.Split(',');
                    String mapIndexIn = mapLineParts[0];
                    String backgroundLocation = "Images/Maps/" + mapLineParts[1];
                    BitmapImage mapBackgroundIn = new BitmapImage(new Uri(backgroundLocation, UriKind.RelativeOrAbsolute));
                    float encounterRateIn = Convert.ToSingle(mapLineParts[3]);
                    String encounterInfoLocation = "GameData/Maps/" + mapLineParts[4];
                    Uri encounterInfoFile = new Uri(encounterInfoLocation, UriKind.RelativeOrAbsolute);
                    Encounter[] encountersIn = LoadEncounterList(encounterInfoFile);


                    String mapInfoLocation = "GameData/Maps/" + mapLineParts[2];
                    //create mapMovegrid using mapInfoLocation file
                    Uri mapInfoFile = new Uri(mapInfoLocation, UriKind.RelativeOrAbsolute);
                    StreamReader mapInfoReader = new StreamReader(App.GetResourceStream(mapInfoFile).Stream);
                    String mapInfoLine = mapInfoReader.ReadLine();
                    bool[,] mapMoveGridIn = new bool[GameMap.MAP_WIDTH, GameMap.MAP_LENGTH];
                    for (int j = GameMap.MAP_LENGTH - 1; j >= 0; j--)
                    {
                        for (int i = 0; i < GameMap.MAP_WIDTH; i++)
                        {
                            if (mapInfoLine[i] == 'X')
                            {
                                mapMoveGridIn[i, j] = false;
                            }
                            else
                            {
                                mapMoveGridIn[i, j] = true;
                            }
                        }
                        mapInfoLine = mapInfoReader.ReadLine();
                    }
                    //create map
                    mapList[mapListArrayPosition] = new GameMap(mapBackgroundIn, mapMoveGridIn, mapIndexIn, encounterRateIn, encountersIn);
                    //add events (if any)
                    
                    while (mapInfoLine != null)
                    {
                        //if next line is not null, there are events to add


                        //split into substrings based on ',' character
                        String[] mapEventParts = mapInfoLine.Split(',');

                        //  the second and third strings are x and y coordinates where the event is located on this map
                        Position eventLocation = new Position(Convert.ToInt32(mapEventParts[1]), Convert.ToInt32(mapEventParts[2]));

                        //if the first is "MM", it shows a map move event
                        if (mapEventParts[0] == "MM")
                        {
                            String mapIDIn = mapEventParts[3];
                            Position eventDestination = new Position(Convert.ToInt32(mapEventParts[4]), Convert.ToInt32(mapEventParts[5]));
                            MapMoveEvent newEventToAdd = new MapMoveEvent(mapIDIn, eventDestination);
                            mapList[mapListArrayPosition].AddMapMoveEvent(newEventToAdd, eventLocation);
                            //  next string is the mapID of the map that is moved to
                            //  next two strings are the x and y coordinates where the character moves on the next map
                        }
                        else if (mapEventParts[0] == "Shop")
                        {
                            String NPCNameToAdd = mapEventParts[3];
                            //Split dialog on | symbols
                            String[] dialogToAdd = mapEventParts[7].Split('|');

                            String portraitName = mapEventParts[4];
                            BitmapImage NPCPortraitToAdd = new BitmapImage(new Uri("Images/" + portraitName, UriKind.RelativeOrAbsolute));

                            //Shop inventory
                            //three possible lines, first is Items, second is Equipment, third is Materials
                            //each line's first member is either I, E or M
                            String shopFileName = mapEventParts[6];
                            Uri shopTextFile = new Uri("GameData/Shops/" + shopFileName, UriKind.RelativeOrAbsolute);
                            StreamReader shopReader = new StreamReader(App.GetResourceStream(shopTextFile).Stream);

                            Item[] shopItemInventory = null;
                            Equipment[] shopEquipmentInventory = null;
                            //Material[] shopMaterialInventory;

                            String shopInfoLine = shopReader.ReadLine();
                            while (shopInfoLine != null)
                            {
                                String[] shopInfoParts = shopInfoLine.Split(',');
                                if (shopInfoParts[0] == "I")
                                {
                                    shopItemInventory = new Item[shopInfoParts.Length - 1];
                                    for (int i = 1; i < shopInfoParts.Length; i++)
                                    {
                                        shopItemInventory[i - 1] = itemList[playerCharacter.GetItemWithName(shopInfoParts[i])];
                                    }
                                }
                                else if (shopInfoParts[0] == "E")
                                {
                                    shopEquipmentInventory = new Equipment[shopInfoParts.Length - 1];
                                    for (int i = 1; i < shopInfoParts.Length; i++)
                                    {
                                        shopEquipmentInventory[i - 1] = equipmentList[playerCharacter.GetEquipmentWithName(shopInfoParts[i])];
                                    }
                                }
                                //else, for Materials
                                //to be implemented later

                                shopInfoLine = shopReader.ReadLine();
                            }


                            //create new event with given information
                            NPCEvent newEventToAdd = new NPCEvent(NPCEventType.Shop, NPCNameToAdd, NPCPortraitToAdd, dialogToAdd);
                            newEventToAdd.SetItemInventory(shopItemInventory);
                            newEventToAdd.SetEquipmentInventory(shopEquipmentInventory);

                            //Set NPC shop's price increase value
                            newEventToAdd.SetShopPriceIncrease(Convert.ToSingle(mapEventParts[5]));

                            //add event to map being loaded
                            mapList[mapListArrayPosition].AddNPCEvent(newEventToAdd, eventLocation);
                        }
                        else if (mapEventParts[0] == "Quest")
                        { 
                            // to be implemented at a later date
                        }
                        else if (mapEventParts[0] == "Dialog")
                        {
                            // to be implemented at a later date
                        }
                        mapInfoLine = mapInfoReader.ReadLine();
                    }
                    //repeat until the next line read is null, to add all events in the file.                     
                    mapLineToLoad = sr.ReadLine();
                    mapListArrayPosition++;
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        /*  LoadEncounterList() takes in a Uri file containing information about encounters on a given map.
         *  It returns an array of Encounter objects, used in creating GameMap objects.
         */
        private Encounter[] LoadEncounterList(Uri fileIn)
        {
            Encounter[] encounterIn = new Encounter[CountLinesInFile(fileIn)];
            StreamReader sr = new StreamReader(App.GetResourceStream(fileIn).Stream);
            String encounterInfoLine = sr.ReadLine();
            for (int i = 0; i < encounterIn.Length; i++)
            {
                String[] encounterInfoParts = encounterInfoLine.Split(',');
                float encounterRateIn = Convert.ToSingle(encounterInfoParts[0]);

                //create array of AICharacters to fight in the encounter, length equal to encounterInfoParts-1, max of 5
                AICharacter[] opponentsInEncounter;
                if (encounterInfoParts.Length <= 6)
                {
                    opponentsInEncounter = new AICharacter[encounterInfoParts.Length - 1];
                }
                else
                {
                    //if encounter file has more than 5 opponents in it, only use the first 5.
                    opponentsInEncounter = new AICharacter[5];
                }
                for (int j = 0; j < opponentsInEncounter.Length; j++)
                {
                    //Add copies of each AICharacter listed in Encounter file, to AICharacter array
                    opponentsInEncounter[j] = GetCopyOfAIFromList(encounterInfoParts[j + 1]);
                }
                encounterIn[i] = new Encounter(opponentsInEncounter, encounterRateIn);
                encounterInfoLine = sr.ReadLine();
            }

            return encounterIn;
        }


        /*  LoadAICharacterList() loads a list of all AICharcter opponents in the game, for the program
         *  to reference when creating new AICharacters.  These are loaded from the resource 
         *  /GameData/AICharacters.txt
         *  AICharacters loaded from the method are placed in the opponentList array which is initalized
         *  to the size of the number of lines in the AICharacters.txt file.
         */
        private void LoadAICharacterList()
        {
            try
            {
                Uri uriTextFile = new Uri("GameData/AICharacters.txt", UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);

                String lineLoaded = sr.ReadLine();
                int AICharacterArrayPosition = 0;
                opponentList = new AICharacter[CountLinesInFile(uriTextFile)];
                while (lineLoaded != null)
                {
                    //split line input into parts to create AICharacter for opponentList
                    String[] lineParts = lineLoaded.Split(',');

                    String AINameIn = lineParts[0];
                    int AIStrIn = Convert.ToInt32(lineParts[1]);
                    int AIStmIn = Convert.ToInt32(lineParts[2]);
                    int AIAglIn = Convert.ToInt32(lineParts[3]);
                    int AIIntIn = Convert.ToInt32(lineParts[4]);
                    int AIFthIn = Convert.ToInt32(lineParts[5]);
                    int AIPrsIn = Convert.ToInt32(lineParts[6]);
                    String AITypeIn = lineParts[7];
                    int AIMinDamageIn = Convert.ToInt32(lineParts[8]);
                    int AIMaxDamageIn = Convert.ToInt32(lineParts[9]);
                    float AIArmorIn = Convert.ToSingle(lineParts[10]);
                    int currencyCarriedIn = Convert.ToInt32(lineParts[11]);
                    //create character
                    if (AITypeIn == "AttackerAI")
                    {
                        opponentList[AICharacterArrayPosition] = new AttackerAI(AINameIn, AIStrIn, AIStmIn, AIAglIn, AIIntIn, AIFthIn, AIPrsIn, currencyCarriedIn);
                    }
                    else if (AITypeIn == "BalancedAI")
                    {
                        opponentList[AICharacterArrayPosition] = new BalancedAI(AINameIn, AIStrIn, AIStmIn, AIAglIn, AIIntIn, AIFthIn, AIPrsIn, currencyCarriedIn);
                    }
                    else if (AITypeIn == "DefenderAI")
                    {
                        opponentList[AICharacterArrayPosition] = new DefenderAI(AINameIn, AIStrIn, AIStmIn, AIAglIn, AIIntIn, AIFthIn, AIPrsIn, currencyCarriedIn);
                    }
                    else if (AITypeIn == "HealerAI")
                    {
                        opponentList[AICharacterArrayPosition] = new HealerAI(AINameIn, AIStrIn, AIStmIn, AIAglIn, AIIntIn, AIFthIn, AIPrsIn, currencyCarriedIn);
                    }
                    else
                    {
                        //default choice in case an error in input
                        opponentList[AICharacterArrayPosition] = new BalancedAI(AINameIn, AIStrIn, AIStmIn, AIAglIn, AIIntIn, AIFthIn, AIPrsIn, currencyCarriedIn);
                    }
                    
                    opponentList[AICharacterArrayPosition].SetDamage(AIMinDamageIn, AIMaxDamageIn);
                    opponentList[AICharacterArrayPosition].SetArmor(AIArmorIn);

                    String AIPortaitIn = lineParts[12];
                    String portraitLocation = "Images/" + AIPortaitIn;
                    opponentList[AICharacterArrayPosition].SetPortrait(new BitmapImage(new Uri(portraitLocation, UriKind.Relative)));

                    //split input string and add each skill individually
                    //the skill string may have multiple skills, separated by the '.' character
                    String[] skillParts = lineParts[13].Split('.');
                    for (int i = 0; i < skillParts.Length; i++)
                    {
                        String skillToAddIn = skillParts[i];
                        opponentList[AICharacterArrayPosition].AddSkill(skillParts[i]);
                    }
                    opponentList[AICharacterArrayPosition].DetermineLevel();

                    //read next line and repeat until all AICharacters are loaded.
                    lineLoaded = sr.ReadLine();
                    AICharacterArrayPosition++;
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        /*  LoadItemList() loads a list of all items existing in the game, for the program to reference when
         *  needed.  These are loaded from the resource /GameData/Items.txt
         *  Items loaded from the method are placed in the itemList array which is initialized to the size
         *  of the number of lines in the Items.txt file.
         */
        private void LoadItemList()
        {
            try
            {
                Uri uriTextFile = new Uri("GameData/Items.txt", UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);

                String itemLineLoaded = sr.ReadLine();
                int itemsArrayPosition = 0;
                itemList = new Item[CountLinesInFile(uriTextFile)];
                while (itemLineLoaded != null)
                {
                    String[] itemLineParts = itemLineLoaded.Split(',');
                    //add item to itemList[]
                    String itemNameIn = itemLineParts[0];
                    int itemValueIn = Convert.ToInt32(itemLineParts[1]);
                    bool isMagicalIn;
                    if (itemLineParts[2] == "false")
                    {
                        isMagicalIn = false;
                    }
                    else
                    {
                        isMagicalIn = true;
                    }
                    bool isDamagingIn;
                    if (itemLineParts[3] == "false")
                    {
                        isDamagingIn = false;
                    }
                    else
                    {
                        isDamagingIn = true;
                    }
                    bool isHealingIn;
                    if (itemLineParts[4] == "false")
                    {
                        isHealingIn = false;
                    }
                    else
                    {
                        isHealingIn = true;
                    }
                    int itemNumValueIn = Convert.ToInt32(itemLineParts[5]);
                    bool isBeneficialIn;
                    if (itemLineParts[6] == "false")
                    {
                        isBeneficialIn = false;
                    }
                    else
                    {
                        isBeneficialIn = true;
                    }
                    bool isUsableIn;
                    if (itemLineParts[7] == "false")
                    {
                        isUsableIn = false;
                    }
                    else
                    {
                        isUsableIn = true;
                    }
                    //status effects
                    int poisonValueIn = Convert.ToInt32(itemLineParts[8]);
                    float pDefenseValueIn = Convert.ToSingle(itemLineParts[9]);
                    float mDefenseValueIn = Convert.ToSingle(itemLineParts[10]);

                    itemList[itemsArrayPosition] = new Item(itemNameIn, itemValueIn, isMagicalIn, isDamagingIn, isHealingIn, itemNumValueIn, isBeneficialIn, isUsableIn, poisonValueIn, pDefenseValueIn, mDefenseValueIn);
                    //get next line and repeat until all Items are loaded.
                    itemLineLoaded = sr.ReadLine();
                    itemsArrayPosition++;
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        /*  LoadEquipmentList() loads a list of all equipment existing in the game, for the program to reference
         *  when needed.  These are loaded from the resource /GameData/Equipment.txt
         *  Equipment loaded from the method are placed in the equipmentList array which is initialized to the
         *  size equal to the number of lines in the Equipment.txt file.
         */
        private void LoadEquipmentList()
        {
            try
            {
                Uri uriTextFile = new Uri("GameData/Equipment.txt", UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);

                String equipLineLoaded = sr.ReadLine();
                int equipArrayPosition = 0;
                equipmentList = new Equipment[CountLinesInFile(uriTextFile)];
                while (equipLineLoaded != null)
                {
                    String[] equipLineParts = equipLineLoaded.Split(',');
                    String equipType = equipLineParts[0];
                    String equipNameIn = equipLineParts[1];
                    int equipValueIn = Convert.ToInt32(equipLineParts[2]);
                    //if the current line is for a weapon, create a Weapon type
                    if (equipType.Equals("Weapon"))
                    {
                        int weaponMinDamageIn = Convert.ToInt32(equipLineParts[3]);
                        int weaponMaxDamageIn = Convert.ToInt32(equipLineParts[4]);
                        bool isTwoHandedIn = false;
                        if (equipLineParts[5] == "true")
                        {
                            isTwoHandedIn = true;
                        }
                        equipmentList[equipArrayPosition] = new Weapon(equipNameIn, equipValueIn, weaponMinDamageIn, weaponMaxDamageIn, isTwoHandedIn);
                    }
                    //if the current line is for a piece of armor, create an Armor type
                    else if (equipType.Equals("Armor"))
                    {
                        float armorProtectionValueIn = Convert.ToSingle(equipLineParts[3]);
                        ArmorType armorTypeIn = GetArmorType(equipLineParts[4]);
                        equipmentList[equipArrayPosition] = new Armor(equipNameIn, equipValueIn, armorProtectionValueIn, armorTypeIn);
                    }
                    else
                    {
                        //to display an error
                        equipmentList[equipArrayPosition] = new Equipment();
                    }

                    //read next line and repeat until all Equipment is loaded
                    equipArrayPosition++;
                    equipLineLoaded = sr.ReadLine();
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        /*  GetArmorType() accepts a String as input that should relate to an existing ArmorType, and returns
         *  that ArmorType value.
         */
        private ArmorType GetArmorType(string typeIn)
        {
            ArmorType typeToReturn;
            if (typeIn.Equals("Body"))
            {
                typeToReturn = ArmorType.Body;
            }
            else if (typeIn.Equals("Shield"))
            {
                typeToReturn = ArmorType.Shield;
            }
            else
            {
                typeToReturn = ArmorType.Body;
            }
            return typeToReturn;
        }

        /*  LoadSkillList() loads a list of all skills existing in the game, for the program to reference when
         *  needed.  These are loaded from the resource /GameData/Skills.txt
         *  Skills loaded from the method are placed in the skillList array, which is initalized to the size
         *  equal to the number of lines in the Skills.txt file
         */
        private void LoadSkillList()
        {  
            try
            {
                Uri uriTextFile = new Uri("GameData/Skills.txt", UriKind.RelativeOrAbsolute);
                StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);
                String skillLineLoaded = sr.ReadLine();
                int skillsArrayPosition = 0;
                skillList = new Skill[CountLinesInFile(uriTextFile)];
                while (skillLineLoaded != null)
                {
                    //split skillLineLoaded into substrings based on delimiter of ,
                    String[] skillLineParts = skillLineLoaded.Split(',');
                    //add skill to skills[]
                    String skillNameIn = skillLineParts[0];
                    int skillCostIn = Convert.ToInt32(skillLineParts[1]);
                    bool isMagicalIn;
                    if (skillLineParts[2] == "false")
                    {
                        isMagicalIn = false;
                    }
                    else
                    {
                        isMagicalIn = true;
                    }
                    bool isDamagingIn;
                    if (skillLineParts[3] == "false")
                    {
                        isDamagingIn = false;
                    }
                    else
                    {
                        isDamagingIn = true;
                    }
                    bool isHealingIn;
                    if (skillLineParts[4] == "false")
                    {
                        isHealingIn = false;
                    }
                    else
                    {
                        isHealingIn = true;
                    }
                    int skillNumValueIn = Convert.ToInt32(skillLineParts[5]);
                    bool isBeneficialIn;
                    if (skillLineParts[6] == "false")
                    {
                        isBeneficialIn = false;
                    }
                    else
                    {
                        isBeneficialIn = true;
                    }
                    //status effects
                    int poisonValueIn = Convert.ToInt32(skillLineParts[7]);
                    float pDefenseValueIn = Convert.ToSingle(skillLineParts[8]);
                    float mDefenseValueIn = Convert.ToSingle(skillLineParts[9]);

                    skillList[skillsArrayPosition] = new Skill(skillNameIn, skillCostIn, isMagicalIn, isDamagingIn, isHealingIn, skillNumValueIn, isBeneficialIn, poisonValueIn, pDefenseValueIn, mDefenseValueIn);
                    //get next line and repeat until all Skills are added
                    skillLineLoaded = sr.ReadLine();
                    skillsArrayPosition++;
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        /*  UpdateInventoryGrid() updates the contents of the InventoryGrid object, specifically displaying
         *  the contents of the ListBoxes relating to Items, Equipment and Materials
         */
        private void UpdateInventoryGrid()
        {
            PopulateItemListBox();
            PopulateEquipmentListBox();
            //PopulateMaterialListBox();
            //display all Materials in InventoryListBox, when Materials are added
        }
        
        /*  PopulateItemListBox() updates the contents of the ItemListBox object in the InventoryGrid,
         *  adding each Item the player possesses to the list in the form of:
         *  (itemQuantity) itemName
         */
        private void PopulateItemListBox()
        {
            ItemListBox.SelectedIndex = -1;
            ItemListBox.Items.Clear();

            String toAddToList;
            //display all Items in ItemListBox
            for (int i = 0; i < itemList.Length; i++)
            {
                if ((itemList[i] != null) && (playerCharacter.HasItem(i)))
                {
                    toAddToList = itemList[i].GetItemName();
                    this.ItemListBox.Items.Add("(" + playerCharacter.GetItemQuantity(i) + ") " + toAddToList);
                }
            }
        }

        /*  PopulateEquipmentListBox() updates the contents of the EquipmentListBox object in the InventoryGrid,
         *  adding each Equipment the player possesses to the list in the form of:
         *  (equipmentQuantity)(E) equipmentName
         *    where (E) is only shown if the item is equipped
         */
        private void PopulateEquipmentListBox()
        {
            EquipmentListBox.SelectedIndex = -1;
            EquipmentListBox.Items.Clear();
            //display all Equipment in EquipmentListBox
            String toAddToList;
            for (int i = 0; i < equipmentList.Length; i++)
            {
                if ((equipmentList[i] != null) && (playerCharacter.HasEquipment(i)))
                {
                    toAddToList = " " + equipmentList[i].GetItemName();
                    //Add (E) after the quantity if the item is equipped currently
                    if (playerCharacter.IsEquipped(equipmentList[i])) 
                    {
                        toAddToList = "(E)" + toAddToList;
                    }
                    this.EquipmentListBox.Items.Add("(" + playerCharacter.GetEquipmentQuantity(i) + ")" + toAddToList);
                }
            }
        }

        /*  PopulateMaterialListBox() updates the contents of the MaterialListBox object in the InventoryGrid,
         *  adding each Material the player possesses to the list in the form of:
         *  (materialQuantity) materialName
         */
        private void PopulateMaterialListBox()
        {
            //display all Equipment in MaterialListBox
            //to be implemented at a later date
        }

        /*  PopulateOutOfCombatSkillComboBox() adds every skill that can be used out of combat, typically healing
         *  and status recovery skills, to the CharacterSkillComboBox to be used outside of combat.
         */
        private void PopulateOutOfCombatSkillComboBox()
        {
            for (int i = 0; i < skillList.Length; i++)
            {
                if ((skillList[i] != null) && (playerCharacter.IsSkillKnown(i)) && (skillList[i].IsUsableOutOfCombat()))
                {
                    String skillToAdd = skillList[i].GetSkillName();
                    if (!this.CharacterSkillComboBox.Items.Contains(skillToAdd))
                    {
                        this.CharacterSkillComboBox.Items.Add(skillToAdd);
                    }
                }
            }
        }

        /*  UpdateMapView() sets the image used for the MapBackground object to the background of the current
         *  map, as well as sets the position for the CharacterMapIcon to the player's current position.
         */
        private void UpdateMapView()
        {
            CharacterMapIcon.Source = playerCharacter.GetIcon();
            currentMap = playerCharacter.GetCurrentMap();
            currentPos = playerCharacter.GetCurrentPosition();

            MapBackground.Source = currentMap.GetBackground();
            Grid.SetRow(CharacterMapIcon, (GameMap.MAP_LENGTH - 1 - currentPos.Y));
            Grid.SetColumn(CharacterMapIcon, (currentPos.X));
        }

        /*  UpdateCharacterGrid() sets the values in CharacterGrid (level, stats, health, etc.) to the current
         *  values the player has.
         */
        private void UpdateCharacterGrid()
        {
            CharacterImage.Source = playerCharacter.GetPortrait();
            CharacterLabel.Text = playerCharacter.GetName();

            StrValueLabel.Text = playerCharacter.GetStrength().ToString();
            StmValueLabel.Text = playerCharacter.GetStamina().ToString();
            AglValueLabel.Text = playerCharacter.GetAgility().ToString();
            IntValueLabel.Text = playerCharacter.GetIntelligence().ToString();
            FthValueLabel.Text = playerCharacter.GetFaith().ToString();
            PrsValueLabel.Text = playerCharacter.GetPersona().ToString();
            LevelLabel.Text = "Level " + playerCharacter.GetLevel();
            HealthValueLabel.Text = playerCharacter.GetCurrentHealth().ToString() + " / " + playerCharacter.GetMaxHealth().ToString();
            MagicValueLabel.Text = playerCharacter.GetCurrentMagic().ToString() + " / " + playerCharacter.GetMaxMagic().ToString();
            ExperienceValueLabel.Text = playerCharacter.GetCurrentExperience().ToString();
            DamageValueLabel.Text = playerCharacter.GetMinDamage() + " - " + playerCharacter.GetMaxDamage();
            ArmorValueLabel.Text = (Convert.ToInt32(playerCharacter.GetArmor() * 100)) + "";

            PlayerCurrencyValue.Text = playerCharacter.GetCurrentCurrencyAmount().ToString();

            PopulateOutOfCombatSkillComboBox();
            UpdateInventoryGrid();
        }

        /*  TriggerCombat() does the preparation for beginning combat with the array of AICharacters passed in.
         *  It sends a message to the ChatMessages panel, switches out of mapMode and calls StartCombat()
         */
        private void TriggerCombat(AICharacter[] encounterIn)
        {
            String combatNotification = "You encounter a ";
            for (int i = 0; i < encounterIn.Length; i++)
            {
                if (i == 0)
                {
                    combatNotification += encounterIn[i].GetName();
                }
                else if (i == encounterIn.Length - 1)
                {
                    combatNotification += " and a " + encounterIn[i].GetName();
                }
                else
                {
                    combatNotification += ", a " + encounterIn[i].GetName();
                }
            }
            combatNotification += "!";
            AddMessageToChatBox(combatNotification, systemMessageColor);
            SwitchMode();
            StartCombat(encounterIn);
        }

        /*  SwitchMode() changes the current mode between map and combat modes, which changes which
         *  view is shown to the user, as well as disables some functions.
         */
        private void SwitchMode()
        {
            //Change from map to combat mode
            if (mapMode == true)
            {
                //Changes visibility of MapGrid and CombatGrid so the combat view is seen
                MapGrid.Visibility = Visibility.Collapsed;
                CombatGrid.Visibility = Visibility.Visible;
                //Disables the button use in the InventoryGrid to prevent the player from changing equipment
                //or dropping items in combat.
                UseItemInventoryButton.IsEnabled = false;
                DropItemButton.IsEnabled = false;
                CharacterSkillComboBox.IsEnabled = false;
                CharacterUseSkillButton.IsEnabled = false;
                mapMode = false;
            }
            //Change from combat to map mode
            else if (mapMode == false)
            {
                //Changes visibility of MapGrid and CombatGrid so the map view is seen
                CombatGrid.Visibility = Visibility.Collapsed;
                MapGrid.Visibility = Visibility.Visible;
                //Enables the button use in the InventoryGrid, which was disabled in combat mode.
                UseItemInventoryButton.IsEnabled = true;
                DropItemButton.IsEnabled = true;
                CharacterSkillComboBox.IsEnabled = true;
                CharacterUseSkillButton.IsEnabled = true;
                mapMode = true;

                //save character after combat
                SavePlayerCharacter();
            }
            UpdateLayout();
        }

        /*  GetMapWithID() takes in a mapID string and returns the GameMap object found in mapList
         *  with a matching mapID.
         */
        private GameMap GetMapWithID(String mapIDIn)
        {
            for (int i = 0; i < mapList.Length; i++)
            {
                if (mapList[i].GetMapID().Equals(mapIDIn))
                {
                    return mapList[i];
                }
            }
            return null;
        }

        /*  MapMove() takes in a MapMoveEvent object and adjusts the playerCharacter's current map and
         *  position to the values specified in the MapMoveEvent, then calls an update on the map view.
         */
        private void MapMove(MapMoveEvent eventIn)
        {
            if (GetMapWithID(eventIn.GetMapIDToMoveTo()) != null)
            {
                playerCharacter.SetCurrentMap(GetMapWithID(eventIn.GetMapIDToMoveTo()));
                currentMap = GetMapWithID(eventIn.GetMapIDToMoveTo());
                playerCharacter.SetCurrentPosition(eventIn.GetPositionToMoveTo());
                currentPos = eventIn.GetPositionToMoveTo();
                UpdateMapView();

                //save after each map move
                SavePlayerCharacter();
            }
        }

        /*  CheckForRandomEncounter() generates a random number and compares it to the encounter rate
         *  of the current map to see if a random combat encounter is triggered.
         *  If so, it selects an encounter from the map's encounter list and calls TriggerCombat()
         */
        private void CheckForRandomEncounter()
        {
            //check to see if random < encounter chance
            //if true, determine which encounter is selected
            if (rnd.NextDouble() < currentMap.GetEncounterRate())
            {
                Encounter[] encounters = currentMap.GetEncounterList();
                AICharacter[] selectedEncounter = SelectEncounterOpponents(encounters);
                TriggerCombat(selectedEncounter);
                //if a random encounter occurs, there is no need to search before camping
                CampButton.IsEnabled = true;
            }
            else
            {
                //save each time the player does not find a random encounter
                SavePlayerCharacter();
            }
        }

        /*  SelectEncounterOpponents() takes in an encounter list and generates a random number to select an
         *  encounter based on the encounter rates given in the encounter list, then returns the
         *  AICharacter array of the selected encounter.
         */
        private AICharacter[] SelectEncounterOpponents(Encounter[] encounterListIn)
        {
            float chance = 0.0f;
            double random = rnd.NextDouble();

            for (int i = 0; i < encounterListIn.Length; i++)
            {
                chance += encounterListIn[i].GetEncounterChance();
                if (random < chance)
                {
                    return encounterListIn[i].GetOpponentList();
                }
            }
            return null;

        }



                        /* MAP EVENTS */

        /*  MoveEvent() is called whenever the player moves the character.  It checks to see if an
         *  event exists at the new location and starts the event if one exists, or checks for a
         *  random encounter if no event exists.
         */
        private void MoveEvent()
        {
            //moving disables camping until searching is done again.
            CampButton.IsEnabled = false;

            //check if an event exists at the new location
            if (currentMap.GetMapEventAtLocation(currentPos) != null)
            {
                //check for event type to trigger it
                if (currentMap.GetMapEventAtLocation(currentPos) is MapMoveEvent)
                {
                    MapMove(currentMap.GetMapMoveEventAtLocation(currentPos));
                }
                else if (currentMap.GetMapEventAtLocation(currentPos) is NPCEvent)
                {
                    //update display and start the NPC event
                    FadeGrid.Visibility = Visibility.Visible;
                    NPCEventGrid.Visibility = Visibility.Visible;
                    StartNPCEvent(currentMap.GetNPCEventAtLocation(currentPos));
                }
            }
            else
            {
                //only trigger random encounter chances when no events are present
                CheckForRandomEncounter();
            }
        }

        /*  StartNPCEvent() is called from MoveEvent() and handles the execution of each type of NPC
         *  Event (Dialog, Quest and Shop).  It updates the display as needed for the event
         *  NOTE:  Currently only the Shop NPCEventType is implemented
         */
        public void StartNPCEvent(NPCEvent eventIn)
        {
            if (eventIn.GetNPCEventType() == NPCEventType.Dialog)
            {
                //to be implemented at a later date
            }
            else if (eventIn.GetNPCEventType() == NPCEventType.Quest)
            {
                //to be implemented at a later date
            }
            else if (eventIn.GetNPCEventType() == NPCEventType.Shop)
            {
                NPCEventBuyButton.Visibility = Visibility.Visible;
                NPCEventSellButton.Visibility = Visibility.Visible;
                NPCEventButton.Visibility = Visibility.Visible;
                NPCEventButton.Content = "Exit";
                NPCEventPortrait.Source = eventIn.GetNPCPortrait();

                //set NPCName label
                String NPCName = eventIn.GetNPCName();
                NPCNameLabel.Text = NPCName;

                //Start dialog
                String[] dialog = eventIn.GetNPCDialog();
                NPCEventDialogBox.Text = dialog[0];
            }
        }

        /*  MoveNorth() adjusts the position of the player one cell North, if possible, then triggers
         *  an event located at the new position, if any exists, or checks for a random encounter.
         */
        private void MoveNorth()
        {
            Position newPos = new Position(currentPos.X, (currentPos.Y + 1));
            if (currentMap.MoveIsValid(newPos))
            {
                currentPos = newPos;
                //update view
                Grid.SetRow(CharacterMapIcon, (GameMap.MAP_LENGTH - 1 - currentPos.Y));
                MoveEvent();
            }
        }

        /*  MoveNorthButton_Click() makes a call to MoveNorth() when the North button is clicked.
         */
        private void MoveNorthButton_Click(object sender, RoutedEventArgs e)
        {
            MoveNorth();
        }

        /*  MoveEast() adjusts the position of the player one cell East, if possible, then triggers
         *  an event located at the new position, if any exists, or checks for a random encounter.
         */
        private void MoveEast()
        {
            Position newPos = new Position((currentPos.X + 1), currentPos.Y);
            if (currentMap.MoveIsValid(newPos))
            {
                currentPos = newPos;
                //update view
                Grid.SetColumn(CharacterMapIcon, (currentPos.X));
                MoveEvent();
            }
        }

        /*  MoveEastButton_Click() makes a call to MoveEast() when the East button is clicked..
         */
        private void MoveEastButton_Click(object sender, RoutedEventArgs e)
        {
            MoveEast();
        }

        /*  MoveSouth() adjusts the position of the player one cell South, if possible, then triggers
         *  an event located at the new position, if any exists, or checks for a random encounter.
         */
        private void MoveSouth()
        {
            Position newPos = new Position(currentPos.X, (currentPos.Y - 1));
            if (currentMap.MoveIsValid(newPos))
            {
                currentPos = newPos;
                //update view
                Grid.SetRow(CharacterMapIcon, (GameMap.MAP_LENGTH - 1 - currentPos.Y));
                MoveEvent();
            }
        }

        /*  MoveSouthButton_Click() makes a call to MoveSouth() when the South button is clicked.
         */
        private void MoveSouthButton_Click(object sender, RoutedEventArgs e)
        {
            MoveSouth();
        }

        /*  MoveWest() adjusts the position of the player one cell West, if possible, then triggers
         *  an event located at the new position, if any exists, or checks for a random encounter.
         */
        private void MoveWest()
        {
            Position newPos = new Position((currentPos.X - 1), currentPos.Y);
            if (currentMap.MoveIsValid(newPos))
            {
                currentPos = newPos;
                //update view
                Grid.SetColumn(CharacterMapIcon, (currentPos.X));
                MoveEvent();
            }
        }

        /*  MoveWestButton_Click() makes a call to MoveWest() when the West button is clicked.
         */
        private void MoveWestButton_Click(object sender, RoutedEventArgs e)
        {
            MoveWest();
        }


        /*  SearchButton_Click() handles the Search function when the Search button is clicked.
         *  It will check for a random encounter, based on the current map's encounter rate,
         *  and trigger combat if one is encountered.
         *  If not, nothing happens, however the area is then marked as clear to Camp until
         *  the player moves.
         *  NOTE:  When Item creation is implemented, searching may reveal materials that can be
         *         harvested.
         */
        private void SearchButton_Click(object sender, RoutedEventArgs e)
        {
            if (rnd.NextDouble() < currentMap.GetEncounterRate())
            {
                //random chance of combat
                TriggerCombat(SelectEncounterOpponents(currentMap.GetEncounterList()));
            }
            else
            {
                AddMessageToChatBox("You found nothing in your search", systemMessageColor);
            }
            //Player may camp after searching.
            CampButton.IsEnabled = true;
        }

        /*  CampButton_Click() is only enabled after the player participates in combat or
         *  uses Search in a space.
         *  Camping will restore a player character to full health and magic, then disable
         *  the Camp button.
         */
        private void CampButton_Click(object sender, RoutedEventArgs e)
        {
            playerCharacter.SetCurrentHealth(playerCharacter.GetMaxHealth());
            playerCharacter.SetCurrentMagic(playerCharacter.GetMaxMagic());
            UpdateCharacterGrid();
            AddMessageToChatBox("You rest for the night and recover your health and magic.", systemMessageColor);
            //after camping once, the player should search again before camping
            CampButton.IsEnabled = false;
        }

        /*  HarvestButton_Click() will be implemented at a later date.
         */
        private void HarvestButton_Click(object sender, RoutedEventArgs e)
        {
            //to be implemented at a later date.
        }

        /*  CharacterUseSkillButton_Click() uses the skill selected in the CharacterSkillComboBox on the player
         *  if the character has enough current magic to use it.  This is for using out-of-combat skills outside
         *  of combat.
         */
        private void CharacterUseSkillButton_Click(object sender, RoutedEventArgs e)
        {
            if (CharacterSkillComboBox.SelectedIndex >= 0)
            {
                //get skill selected, if any
                Skill selectedSkill = skillList[playerCharacter.GetSkillWithName(CharacterSkillComboBox.SelectedValue.ToString())];
                //compare cost of skill to current magic
                if (selectedSkill.GetSkillCost() <= playerCharacter.GetCurrentMagic())
                {
                    //use skill on self
                    AddMessageToChatBox(UseSkill(selectedSkill, playerCharacter, playerCharacter), systemMessageColor);
                    UpdateCharacterGrid();
                }
                else
                {
                    AddMessageToChatBox("You do not have enough magic to use " + selectedSkill.GetSkillName(), systemMessageColor);
                }
            }
        }

                        /* COMBAT EVENTS */
        /*  FightButton_Click() sets the current playerCommand to "F" for use in combat.
         */
        private void FightButton_Click(object sender, RoutedEventArgs e)
        {
            playerCommand = "F";
        }

        /*  DefendButton_Click() sets the current playerCommand to "D" for use in combat.
         */
        private void DefendButton_Click(object sender, RoutedEventArgs e)
        {
            playerCommand = "D";
        }

        /*  SkillButton_Click() sets the current playerCommand to "S" for use in combat.
         */
        private void SkillButton_Click(object sender, RoutedEventArgs e)
        {
            playerCommand = "S";
        }

        /*  ItemButton_Click() sets the current playerCommand to "I" for use in combat.
         */
        private void ItemButton_Click(object sender, RoutedEventArgs e)
        {
            playerCommand = "I";
        }

        /*  RunButton_Click() sets the current playerCommand to "R" for use in combat.
         */
        private void RunButton_Click(object sender, RoutedEventArgs e)
        {
            playerCommand = "R";
        }

        /*  SkillComboBox_SelectionChanged() updates the SkillCostLabel to the magic cost of the selected Skill
         */
        private void SkillComboBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            Skill selectedSkill = skillList[playerCharacter.GetSkillWithName(SkillComboBox.SelectedValue.ToString())];
            SkillCostLabel.Text = selectedSkill.GetSkillCost().ToString();
        }

        /*  ItemComboBox_SelectionChanged() updates the ItemQuantityLabel to the quantity of the selected Item
         *  the player has.
         */
        private void ItemComboBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            int selectedItemNum = playerCharacter.GetItemWithName(ItemComboBox.SelectedValue.ToString());
            int quantity = playerCharacter.GetItemQuantity(selectedItemNum);
            ItemQuantityLabel.Text = quantity.ToString();
        }

        /*  The following four methods are used to select opponents in combat, ignoring clicks on opponents
         *  that are not alive.
         */
        private void SubCompACombatPortrait_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            //make this opponent the currently selected one, if it is alive
            if ((selectedCompPosition == 0) && (compTeam[1].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 1;
                selectedComp = compTeam[selectedCompPosition];
            }
            else if ((selectedCompPosition > 0) && (compTeam[0].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 0;
                selectedComp = compTeam[selectedCompPosition];
            }
            UpdateCombatDisplay();
        }

        private void SubCompBCombatPortrait_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            if ((selectedCompPosition <= 1) && (compTeam[2].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 2;
                selectedComp = compTeam[selectedCompPosition];
            }
            else if ((selectedCompPosition > 1) && (compTeam[1].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 1;
                selectedComp = compTeam[selectedCompPosition];
            }
            UpdateCombatDisplay();
        }

        private void SubCompCCombatPortrait_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            if ((selectedCompPosition <= 2) && (compTeam[3].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 3;
                selectedComp = compTeam[selectedCompPosition];
            }
            else if ((selectedCompPosition > 2) && (compTeam[2].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 2;
                selectedComp = compTeam[selectedCompPosition];
            }
            UpdateCombatDisplay();
        }

        private void SubCompDCombatPortrait_MouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            if ((selectedCompPosition <= 3) && (compTeam[4].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 4;
                selectedComp = compTeam[selectedCompPosition];
            }
            else if ((selectedCompPosition > 3) && (compTeam[3].GetCurrentHealth() > 0))
            {
                selectedCompPosition = 3;
                selectedComp = compTeam[selectedCompPosition];
            }
            UpdateCombatDisplay();
        }



                        /*  INVENTORY EVENTS  */
        /*  ItemListBox_SelectionChanged() changes the content of InventorySelectionDetails when an Item
         *  is selected in ItemListBox, to give information about the Item selected.
         */
        private void ItemListBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (ItemListBox.SelectedIndex >= 0)
            {
                String itemName = ItemListBox.SelectedValue.ToString();
                //change selection in ItemComboBox
                if (mapMode == false)
                {
                    ItemComboBox.SelectedValue = ItemListBox.SelectedValue;
                }
                Item itemSelected = itemList[playerCharacter.GetItemWithName(itemName.Substring(itemName.IndexOf(' ') + 1))];
                String messageToDisplay = itemSelected.GetItemName() + "\n";
                messageToDisplay += itemSelected.GetItemValue() + " gold coins \n";
                messageToDisplay += GetSelectedItemDescription(itemSelected);

                InventorySelectionDetails.Text = messageToDisplay;
            }
            else
            {
                InventorySelectionDetails.Text = "";
            }
        }
        
        /*  GetSelectedItemDescription() returns a string containing information about the given Item
         *  object, including its name, value and whatever effects it has.
         */
        private String GetSelectedItemDescription(Item itemSelected)
        {
            String messageToDisplay = "";
            if (itemSelected.IsDamaging())
            {
                messageToDisplay += "Deals around " + itemSelected.GetNumValue() + " points of damage \n";
            }
            if (itemSelected.IsHealing())
            {
                messageToDisplay += "Heals around " + itemSelected.GetNumValue() + " health \n";
            }
            if (itemSelected.GetPoisonEffect() != 0)
            {
                if (itemSelected.GetPoisonEffect() > 0)
                {
                    messageToDisplay += "May inflict poison \n";
                }
                else
                {
                    messageToDisplay += "Removes poison \n";
                }
            }
            if (itemSelected.GetPDefenseChange() != 0)
            {
                if (itemSelected.GetPDefenseChange() > 0)
                {
                    messageToDisplay += "Increases physical defense by " + itemSelected.GetPDefenseChange() * 100 + "% \n";
                }
                else
                {
                    messageToDisplay += "May decrease physical defense by " + itemSelected.GetPDefenseChange() * -100 + "% \n";
                }
            }
            if (itemSelected.GetMDefenseChange() != 0)
            {
                if (itemSelected.GetMDefenseChange() > 0)
                {
                    messageToDisplay += "Increases magical defense by " + itemSelected.GetMDefenseChange() * 100 + "% \n";
                }
                else
                {
                    messageToDisplay += "May decrease magical defense by " + itemSelected.GetMDefenseChange() * -100 + "% \n";
                }
            }
            if (itemSelected.IsMagic())
            {
                messageToDisplay += "(Magical)";
            }

            return messageToDisplay;
        }

        /*  EquipmentListBox_SelectionChanged() changes the content of InventorySelectionDetails when an
         *  Equipment object is selected in EquipmentListBox, to give information about the object selected.
         *  The UseItemInventoryButton is changed to reflect the equipped status of the selected object.
         */
        private void EquipmentListBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (EquipmentListBox.SelectedIndex >= 0)
            {
                String equipmentName = EquipmentListBox.SelectedValue.ToString();
                //change UseItemInventoryButton
                if (equipmentName.Contains("(E)"))
                {
                    UseItemInventoryButton.Content = "Unequip";
                }
                else
                {
                    UseItemInventoryButton.Content = "Equip";
                }
                Equipment equipmentSelected = equipmentList[playerCharacter.GetEquipmentWithName(equipmentName.Substring(equipmentName.IndexOf(' ') + 1))];
                String messageToDisplay = equipmentSelected.GetItemName() + "\n";
                messageToDisplay += equipmentSelected.GetItemValue() + " gold coins \n";
                messageToDisplay += GetSelectedEquipmentDescription(equipmentSelected);

                InventorySelectionDetails.Text = messageToDisplay;
            }
            else
            {
                InventorySelectionDetails.Text = "";
            }
        }

        /*  GetSelectedEquipmentDescription() returns a string containing information about the given
         *  Equipment object, including its name, value and damage or armor values.
         */
        private String GetSelectedEquipmentDescription(Equipment equipmentSelected)
        {
            String messageToDisplay = "";
            if (equipmentSelected is Weapon)
            {
                Weapon weaponSelected = (Weapon)equipmentSelected;
                messageToDisplay += "Damage: " + weaponSelected.GetWeaponMinDamage() + "-" + weaponSelected.GetWeaponMaxDamage() + "\n";
                if (weaponSelected.IsTwoHanded())
                {
                    messageToDisplay += "Two-Handed";
                }
            }
            else if (equipmentSelected is Armor)
            {
                Armor armorSelected = (Armor)equipmentSelected;
                messageToDisplay += "Armor: " + armorSelected.GetProtectionValue() * 100 + "\n";
                messageToDisplay += armorSelected.GetArmorType().ToString();
            }
            return messageToDisplay;
        }

        /*  EquipmentTab_GotFocus() changes the contents of the UseItemInventoryButton to "Equip" or
         *  "Unequip" when the Equipment tab gets focus.
         */
        private void EquipmentTab_GotFocus(object sender, RoutedEventArgs e)
        {
            UseItemInventoryButton.Content = "Equip";
            String equipmentName = EquipmentListBox.SelectedValue.ToString();
            if (equipmentName.Contains("(E)"))
            {
                UseItemInventoryButton.Content = "Unequip";
            }
            else
            {
                UseItemInventoryButton.Content = "Equip";
            }
        }

        /*  ItemTab_GotFocus() changes the contents of the UseItemInventory to "Use Item" when the Item
         *  tab gets focus.
         */
        private void ItemTab_GotFocus(object sender, RoutedEventArgs e)
        {
            UseItemInventoryButton.Content = "Use Item";
        }

        /*  UseItemInventoryButton_Click() performs an action based on the current contents of UseItemInventoryButton.
         *  "Use Item" will use the item selected in ItemListBox on the player.
         *  "Equip" will call Equip() on the selected object in EquipmentListBox.
         *  "Unequip" will call Unequip() on the selected object in EquipmentListBox.
         */
        private void UseItemInventoryButton_Click(object sender, RoutedEventArgs e)
        {
            String buttonType = UseItemInventoryButton.Content.ToString();
            if (buttonType == "Use Item")
            {
                String selectedItemName = ItemListBox.SelectedValue.ToString();
                int itemIndex = playerCharacter.GetItemWithName(selectedItemName.Substring(selectedItemName.IndexOf(' ') + 1));
                Item itemSelected = itemList[itemIndex];
                //if Item is usable out of combat, use on self
                if ((playerCharacter.GetItemQuantity(itemIndex) > 0) && (itemSelected.IsUsableOutOfCombat()))
                {
                    String messageDisplayed = UseItem(itemSelected, playerCharacter);
                    playerCharacter.RemoveItem(itemIndex);
                    AddMessageToChatBox(messageDisplayed, systemMessageColor);
                }
            }
            else if (buttonType == "Equip")
            {
                String selectedEquipmentName = EquipmentListBox.SelectedValue.ToString();
                int equipmentIndex = playerCharacter.GetEquipmentWithName(selectedEquipmentName.Substring(selectedEquipmentName.IndexOf(' ') + 1));
                Equipment equipmentSelected = equipmentList[equipmentIndex];
                playerCharacter.Equip(equipmentSelected);
            }
            else if (buttonType == "Unequip")
            {
                String selectedEquipmentName = EquipmentListBox.SelectedValue.ToString();
                int equipmentIndex = playerCharacter.GetEquipmentWithName(selectedEquipmentName.Substring(selectedEquipmentName.IndexOf(' ') + 1));
                Equipment equipmentSelected = equipmentList[equipmentIndex];
                playerCharacter.Unequip(equipmentSelected);
            }
            //save after each inventory manipulation
            SavePlayerCharacter();
            UpdateCharacterGrid();
        }

        /*  DropItemButton_Click() allows the player to drop an item they possess in their inventory.  It
         *  deducts one of the selected item from the player's inventory each time the button is pressed,
         *  updating the view each time (which removes items once the number they possess reaches zero).
         *  It does not allow the player to drop the last piece of an equipped piece of equipment.
         *  
         *  Dropped items are lost forever.
         */
        private void DropItemButton_Click(object sender, RoutedEventArgs e)
        {
            //if selected tab is Items
            if (InventoryTabControl.SelectedItem == ItemTab)
            {
                //get name of selected item from ItemListBox
                if (ItemListBox.SelectedIndex >= 0)
                {
                    String selectedItemName = ItemListBox.SelectedValue.ToString();
                    selectedItemName = selectedItemName.Substring(selectedItemName.IndexOf(' ') + 1);
                    Item selectedItem = itemList[playerCharacter.GetItemWithName(selectedItemName)];

                    //check to make sure player has at least one of the item to be dropped
                    //if (playerCharacter.GetItemQuantity(playerCharacter.GetItemWithName(selectedItem.GetItemName())) > 0)
                    if (playerCharacter.GetItemQuantity(selectedItem) > 0)
                    {
                        playerCharacter.RemoveItem(playerCharacter.GetItemWithName(selectedItem.GetItemName()));
                        AddMessageToChatBox("Dropped a " + selectedItem.GetItemName(), systemMessageColor);
                        UpdateInventoryGrid();
                    }
                }
            }
            //if selected tab is Equipment
            else if (InventoryTabControl.SelectedItem == EquipmentTab)
            {
                if (EquipmentListBox.SelectedIndex >= 0)
                {
                    String selectedEquipmentName = EquipmentListBox.SelectedValue.ToString();
                    selectedEquipmentName = selectedEquipmentName.Substring(selectedEquipmentName.IndexOf(' ') + 1);
                    Equipment selectedEquipment = equipmentList[playerCharacter.GetEquipmentWithName(selectedEquipmentName)];
                    //check to make sure the player has at least one of the equipment to be dropped.
                    if (playerCharacter.GetEquipmentQuantity(selectedEquipment) > 0)
                    {
                        //if quantity is 1, check to make sure the equipment is not equipped before dropping
                        if ((playerCharacter.GetEquipmentQuantity(selectedEquipment) == 1) && (playerCharacter.IsEquipped(selectedEquipment)))
                        {
                            AddMessageToChatBox("You can't drop an equipped item", systemMessageColor);
                        }
                        else
                        {
                            playerCharacter.RemoveEquipment(playerCharacter.GetEquipmentWithName(selectedEquipment.GetItemName()));
                            AddMessageToChatBox("Dropped a " + selectedEquipment.GetItemName(), systemMessageColor);
                            UpdateInventoryGrid();
                        }
                    }
                }
                //do not allow dropping of last equipped item

            }
        }

                        /*  WELCOME SCREEN EVENTS */
        /*  LoadCharacterButton_Click() attempts to load a saved character from Isolated Storage by calling
         *  LoadPlayerCharacter().  If the load is successful, it changes the view from the welcome screen
         *  to the map view.
         */
        private void LoadCharacterButton_Click(object sender, RoutedEventArgs e)
        {
            if (LoadPlayerCharacter() == true)
            {
                StartGrid.Visibility = Visibility.Collapsed;
                UpdateCharacterGrid();
                UpdateMapView();
            }
        }

        /*  NewCharacterButton_Click() makes a call to CreateNewCharacter().
         */
        private void NewCharacterButton_Click(object sender, RoutedEventArgs e)
        {
            LoadCreateNewCharacterScreen();
        }

        /*  CreateNewCharacter() changes the current view to the NewCharacterGrid and initializes
         *  the combo boxes used for choosing character classes and portraits.
         */
        private void LoadCreateNewCharacterScreen()
        {
            StartGrid.Visibility = Visibility.Collapsed;
            NewCharacterGrid.Visibility = Visibility.Visible;

            //populate NewCharacterClassSelectionComboBox with contents of PlayerClasses.txt
            Uri uriTextFile = new Uri("GameData/PlayerClasses.txt", UriKind.RelativeOrAbsolute);
            StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);
            String currentLineRead = sr.ReadLine();
            while (currentLineRead != null)
            {
                NewCharacterClassSelectionComboBox.Items.Add(currentLineRead);
                currentLineRead = sr.ReadLine();
            }
            //populare NewCharacterPortraitSelectionComboBox with contents of CharacterPortraits.txt
            uriTextFile = new Uri("GameData/CharacterPortraits.txt", UriKind.RelativeOrAbsolute);
            sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);
            currentLineRead = sr.ReadLine();
            while (currentLineRead != null)
            {
                NewCharacterPortraitSelectionComboBox.Items.Add(currentLineRead);
                currentLineRead = sr.ReadLine();
            }
        }

        /*  NewCharacterClassSelectionComboBox_SelectionChanged() displays the starting stats of
         *  the character class selected in the NewCharacterClassSelectionComboBox on the new
         *  character creation screen.
         */
        private void NewCharacterClassSelectionComboBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            String messageDisplayed = "";

            //load ClassLevelTemplate.txt of the appropriate class
            Uri uriTextFile;
            if (NewCharacterClassSelectionComboBox.SelectedValue.ToString().Equals("Fighter"))
            {
                uriTextFile = new Uri("GameData/FighterLevelTemplate.txt", UriKind.RelativeOrAbsolute);
            }
            else if (NewCharacterClassSelectionComboBox.SelectedValue.ToString().Equals("Rogue"))
            {
                uriTextFile = new Uri("GameData/RogueLevelTemplate.txt", UriKind.RelativeOrAbsolute);
            }
            else if (NewCharacterClassSelectionComboBox.SelectedValue.ToString().Equals("Cleric"))
            {
                uriTextFile = new Uri("GameData/ClericLevelTemplate.txt", UriKind.RelativeOrAbsolute);
            }
            else if (NewCharacterClassSelectionComboBox.SelectedValue.ToString().Equals("Wizard"))
            {
                uriTextFile = new Uri("GameData/WizardLevelTemplate.txt", UriKind.RelativeOrAbsolute);
            }
            else
            {
                uriTextFile = new Uri("GameData/FighterLevelTemplate.txt", UriKind.RelativeOrAbsolute);
            }
            StreamReader sr = new StreamReader(App.GetResourceStream(uriTextFile).Stream);
            String classLineRead = sr.ReadLine();
            //the first line of a ClassLevelTemplate file holds the starting stats of the class
            //the string will be in the form of level,strength,stamina,agility,intelligence,faith,persona,skillsLearned
            String[] classStartParts = classLineRead.Split(',');

            NewCharacterStrengthValue.Text = classStartParts[1];
            NewCharacterStaminaValue.Text = classStartParts[2];
            NewCharacterAgilityValue.Text = classStartParts[3];
            NewCharacterIntelligenceValue.Text = classStartParts[4];
            NewCharacterFaithValue.Text = classStartParts[5];
            NewCharacterPersonaValue.Text = classStartParts[6];

            NewCharacterFirstSkillValue.Text = "Starts with the " + classStartParts[7] + " skill.";

            NewCharacterClassDescription.Text = messageDisplayed;
        }

        /*  NewCharacterPortraitSelectionComboBox_SelectionChanged() displays an image of the
         *  selected portrait in the new character creation screen.
         */
        private void NewCharacterPortraitSelectionComboBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            String selectedPortrait = NewCharacterPortraitSelectionComboBox.SelectedValue.ToString();
            Uri portraitImageFile = new Uri("Images/" + selectedPortrait, UriKind.RelativeOrAbsolute);
            BitmapImage portraitDisplay = new BitmapImage(portraitImageFile);

            NewCharacterPortrait.Source = portraitDisplay;
        }

        /*  NewCharacterCreateButton_Click() checks to make sure the player has entered a character
         *  name and selected both a class and portrait.  If so, it creates a new character with the
         *  given information and changes to the initial map view.  If not, it displays an error on
         *  the new character creation page to alert the player to add additional information.
         */
        private void NewCharacterCreateButton_Click(object sender, RoutedEventArgs e)
        {
            String errorMessage = "";
            Boolean errorsExist = false;
            if (NewCharacterNameEntryBox.Text.ToString().Trim() == "")
            {
                errorMessage += "Please enter a name for the character." + "\n";
                errorsExist = true;
            }
            if (NewCharacterClassSelectionComboBox.SelectedIndex == -1)
            {
                errorMessage += "Please select a class." + "\n";
                errorsExist = true;
            }
            if (NewCharacterPortraitSelectionComboBox.SelectedIndex == -1)
            {
                errorMessage += "Please select a portrait." + "\n";
                errorsExist = true;
            }
            NewCharacterErrorMessageBox.Text = errorMessage;

            if (errorsExist == false)
            {
                //create character
                String classSelection = NewCharacterClassSelectionComboBox.SelectedValue.ToString();
                PlayerCharacter newCharacter = CreateNewCharacter(classSelection);
                if (newCharacter != null)
                {
                    playerCharacter = newCharacter;

                    //save character after initial character creation
                    SavePlayerCharacter();

                    NewCharacterGrid.Visibility = Visibility.Collapsed;
                    UpdateCharacterGrid();
                    UpdateMapView();
                }
            }
        }

        /*  CreateNewCharacter() takes in a string with the given character class name and returns a
         *  PlayerCharacter object of a first level instance of that class.
         */
        private PlayerCharacter CreateNewCharacter(String classNameIn)
        {
            Uri classTextFileIn = null;
            //set characterClass to default value
            PlayerClass characterClass = PlayerClass.Fighter;
            if (classNameIn == "Fighter")
            {
                classTextFileIn = new Uri("GameData/FighterLevelTemplate.txt", UriKind.RelativeOrAbsolute);
                characterClass = PlayerClass.Fighter;
            }
            else if (classNameIn == "Rogue")
            {
                classTextFileIn = new Uri("GameData/RogueLevelTemplate.txt", UriKind.RelativeOrAbsolute);
                characterClass = PlayerClass.Rogue;
            }
            else if (classNameIn == "Cleric")
            {
                classTextFileIn = new Uri("GameData/ClericLevelTemplate.txt", UriKind.RelativeOrAbsolute);
                characterClass = PlayerClass.Cleric;
            }
            else if (classNameIn == "Wizard")
            {
                classTextFileIn = new Uri("GameData/WizardLevelTemplate.txt", UriKind.RelativeOrAbsolute);
                characterClass = PlayerClass.Wizard;
            }

            if (classTextFileIn != null)
            {
                StreamReader sr = new StreamReader(App.GetResourceStream(classTextFileIn).Stream);
                String classLineIn = sr.ReadLine();
                //load class beginning stats and skill to create the new character
                String[] classLineParts = classLineIn.Split(',');
                int newCharacterStr = Convert.ToInt32(classLineParts[1]);
                int newCharacterStm = Convert.ToInt32(classLineParts[2]);
                int newCharacterAgl = Convert.ToInt32(classLineParts[3]);
                int newCharacterInt = Convert.ToInt32(classLineParts[4]);
                int newCharacterFth = Convert.ToInt32(classLineParts[5]);
                int newCharacterPrs = Convert.ToInt32(classLineParts[6]);
                String characterName = NewCharacterNameEntryBox.Text.ToString();
                //PlayerCharacter newCharacter = new PlayerCharacter(characterName, characterClass, 1, newCharacterStr, newCharacterStm, newCharacterAgl, newCharacterInt, newCharacterFth, newCharacterPrs, 0);
                PlayerCharacter newCharacter = new PlayerCharacter(characterName, characterClass, newCharacterStr, newCharacterStm, newCharacterAgl, newCharacterInt, newCharacterFth, newCharacterPrs);
                
                BitmapImage characterImageIn = new BitmapImage(new Uri("Images/" + NewCharacterPortraitSelectionComboBox.SelectedValue.ToString(), UriKind.RelativeOrAbsolute));
                newCharacter.SetPortrait(characterImageIn);
                newCharacter.SetIcon(characterImageIn);

                //add first level skill
                newCharacter.AddSkill(newCharacter.GetSkillWithName(classLineParts[7]));

                //add initial equipment
                Uri initialEquipmentFile = new Uri("GameData/NewCharacterEquipment.txt", UriKind.RelativeOrAbsolute);
                sr = new StreamReader(App.GetResourceStream(initialEquipmentFile).Stream);
                String initialEquipmentLine = sr.ReadLine();
                while (initialEquipmentLine != null)
                {
                    String[] initialEquipmentParts = initialEquipmentLine.Split(',');
                    if (initialEquipmentParts[0].ToString().Equals(classNameIn))
                    {
                        for (int i = 1; i < initialEquipmentParts.Length; i++)
                        {
                            newCharacter.AddEquipment(newCharacter.GetEquipmentWithName(initialEquipmentParts[i]), 1);
                            newCharacter.Equip(equipmentList[newCharacter.GetEquipmentWithName(initialEquipmentParts[i])]);
                        }
                    }
                    initialEquipmentLine = sr.ReadLine();
                }

                //set map and location
                newCharacter.SetCurrentMap(mapList[0]);
                newCharacter.SetCurrentPosition(new Position(7, 4));

                return newCharacter;
            }
            else
            {
                return null;
            }

        }

                        /*  NPCEVENT EVENTS  */

        /*  NPCEventButton_Click() handles the use of the button found in all NPCEvents.  The button is
         *  typically labeled "Exit", which will close the NPCEvent view, but during multi-line dialogs,
         *  the button may be labeled "Next" to move to the next line.
         *  NOTE:  This will be implemented when Dialog NPCEvents are completed
         */
        private void NPCEventButton_Click(object sender, RoutedEventArgs e)
        {
            if (NPCEventButton.Content.ToString() == "Exit")
            {
                FadeGrid.Visibility = Visibility.Collapsed;
                NPCEventGrid.Visibility = Visibility.Collapsed;
            }
            else if (NPCEventButton.Content.ToString() == "Next")
            {
                //to be implemented at a later date.
            }
        }

        /*  NPCEventBuyButton_Click() opens up the NPC shop box used for buying items and populates
         *  it with the NPC's available inventory.
         *  If an NPC does not sell any items of a specific type (Item, Equipment or Material) the
         *  player will not be able to select the related tab.
         *  Player inventory manipulation is disabled at this time.
         */
        private void NPCEventBuyButton_Click(object sender, RoutedEventArgs e)
        {
            //Buy-specific labels and button names
            NPCShopBuySellButton.Content = "Buy";
            NPCShopCostLabel.Text = "Cost:";

            //Show NPCEventBuySellGrid
            NPCShopGrid.Visibility = Visibility.Visible;
            NPCEventGrid.Visibility = Visibility.Collapsed;
            UseItemInventoryButton.IsEnabled = false;
            DropItemButton.IsEnabled = false;

            //populate various lists
            Item[] itemInventory = currentMap.GetNPCEventAtLocation(currentPos).GetItemInventory();
            for (int i = 0; i < itemInventory.Length; i++)
            {
                NPCShopItemsListBox.Items.Add(itemInventory[i].GetItemName());
            }

            Equipment[] equipmentInventory = currentMap.GetNPCEventAtLocation(currentPos).GetEquipmentInventory();
            for (int i = 0; i < equipmentInventory.Length; i++)
            {
                NPCShopEquipmentListBox.Items.Add(equipmentInventory[i].GetItemName());
            }
            
            //if list is empty, disable tab
            if (NPCShopItemsListBox.Items.Count == 0)
            {
                NPCShopItemTab.IsEnabled = false;
            }
            else
            {
                NPCShopItemTab.IsEnabled = true;
            }
            if (NPCShopEquipmentListBox.Items.Count == 0)
            {
                NPCShopEquipmentTab.IsEnabled = false;
            }
            else
            {
                NPCShopEquipmentTab.IsEnabled = true;
            }
            if (NPCShopMaterialsListBox.Items.Count == 0)
            {
                NPCShopMaterialsTab.IsEnabled = false;
            }
            else
            {
                NPCShopMaterialsTab.IsEnabled = true;
            }
        }

        /*  NPCEventSellButton_Click() opens up the NPC shop box used for selling items and populates
         *  it with the player's available inventory.
         *  If the player does not have any items of a specific type (Item, Equipment or Material),
         *  the player will not be able to select the related tab.
         *  Player inventory manipulation is disabled at this time.
         */
        private void NPCEventSellButton_Click(object sender, RoutedEventArgs e)
        {
            //Sell-specific labels and button names
            NPCShopBuySellButton.Content = "Sell";
            NPCShopCostLabel.Text = "Value: ";
            //show NPCEventBuySellGrid
            NPCShopGrid.Visibility = Visibility.Visible;
            NPCEventGrid.Visibility = Visibility.Collapsed;
            UseItemInventoryButton.IsEnabled = false;
            DropItemButton.IsEnabled = false;

            //populate lists with player's inventory
            int[] playerItemInventory = playerCharacter.GetItemInventory();
            for (int i = 0; i < playerItemInventory.Length; i++)
            {
                if (playerItemInventory[i] > 0)
                {
                    NPCShopItemsListBox.Items.Add(itemList[i].GetItemName());
                }
            }
            int[] playerEquipmentInventory = playerCharacter.GetEquipmentInventory();
            for (int i = 0; i < playerEquipmentInventory.Length; i++)
            {
                if (playerEquipmentInventory[i] > 0)
                {
                    NPCShopEquipmentListBox.Items.Add(equipmentList[i].GetItemName());
                }
            }

            //if list is empty, disable tab
            if (NPCShopItemsListBox.Items.Count == 0)
            {
                NPCShopItemTab.IsEnabled = false;
            }
            else
            {
                NPCShopItemTab.IsEnabled = true;
            }
            if (NPCShopEquipmentListBox.Items.Count == 0)
            {
                NPCShopEquipmentTab.IsEnabled = false;
            }
            else
            {
                NPCShopEquipmentTab.IsEnabled = true;
            }
            if (NPCShopMaterialsListBox.Items.Count == 0)
            {
                NPCShopMaterialsTab.IsEnabled = false;
            }
            else
            {
                NPCShopMaterialsTab.IsEnabled = true;
            }
        }

        /*  NPCShopExitButton_Click() will close the NPC shop box used for buying or selling items,
         *  returning the player to the previous NPCEvent view.
         *  Player inventory manipulation is enabled again after exiting the shop box.
         */
        private void NPCShopExitButton_Click(object sender, RoutedEventArgs e)
        {
            NPCShopGrid.Visibility = Visibility.Collapsed;
            NPCEventGrid.Visibility = Visibility.Visible;
            UseItemInventoryButton.IsEnabled = true;
            DropItemButton.IsEnabled = true;
            //unselect selected items
            NPCShopItemDescription.Text = "";
            NPCShopCostValue.Text = "0";
            NPCShopItemsListBox.SelectedIndex = -1;
            NPCShopEquipmentListBox.SelectedIndex = -1;
            //clear listboxes
            NPCShopItemsListBox.Items.Clear();
            NPCShopEquipmentListBox.Items.Clear();
        }

        /*  NPCShopItemsListBox_SelectionChanged() changes the item description in the NPC shop box
         *  to the description of the selected Item, as well as changes the value label of the item
         *  to the appropriate price, depending on if the player is Buying or Selling.
         */
        private void NPCShopItemsListBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (NPCShopItemsListBox.SelectedIndex >= 0)
            {
                Item selectedItem = itemList[playerCharacter.GetItemWithName(NPCShopItemsListBox.SelectedValue.ToString())];
                String messageToDisplay = GetSelectedItemDescription(selectedItem);
                NPCShopItemDescription.Text = messageToDisplay;

                if (NPCShopBuySellButton.Content.ToString() == "Buy")
                {
                    int itemCost = Convert.ToInt32(((float)selectedItem.GetItemValue() * currentMap.GetNPCEventAtLocation(currentPos).GetShopPriceIncrease()));
                    NPCShopCostValue.Text = itemCost.ToString();
                }
                else if (NPCShopBuySellButton.Content.ToString() == "Sell")
                {
                    NPCShopCostValue.Text = selectedItem.GetItemValue().ToString();
                }
            }

        }

        /*  NPCShopEquipmentListBox_SelectionChanged() changes the item description in the NPC shop box
         *  to the description of the selected Equipment, as well as changes the value label of the item
         *  to the appropriate price, depending on if the player is Buying or Selling.
         */
        private void NPCShopEquipmentListBox_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            if (NPCShopEquipmentListBox.SelectedIndex >= 0)
            {
                Equipment selectedEquipment = equipmentList[playerCharacter.GetEquipmentWithName(NPCShopEquipmentListBox.SelectedValue.ToString())];
                String messageToDisplay = GetSelectedEquipmentDescription(selectedEquipment);
                NPCShopItemDescription.Text = messageToDisplay;
                if (NPCShopBuySellButton.Content.ToString() == "Buy")
                {
                    int equipmentCost = Convert.ToInt32(((float)selectedEquipment.GetItemValue() * currentMap.GetNPCEventAtLocation(currentPos).GetShopPriceIncrease()));
                    NPCShopCostValue.Text = equipmentCost.ToString();
                }
                else if (NPCShopBuySellButton.Content.ToString() == "Sell")
                {
                    NPCShopCostValue.Text = selectedEquipment.GetItemValue().ToString();
                }
            }
        }

        /*  NPCShopBuySellButton_Click() handles the buying and selling of items in an NPC shop, checking
         *  for appropriate conditions before proceeding, such as checking to see if the player has enough
         *  gold before allowing a purchase, or checking to see if the player has an item before allowing
         *  it to be sold.
         */
        private void NPCShopBuySellButton_Click(object sender, RoutedEventArgs e)
        {
            //if player is Buying items from the store
            if (NPCShopBuySellButton.Content.ToString() == "Buy")
            {
                //check to see which tab is active (Items, Equipment, Materials)
                if (NPCShopTabControl.SelectedItem == NPCShopItemTab)
                {
                    //only attempt to Buy if an Item is selected.
                    if (NPCShopItemsListBox.SelectedIndex >= 0)
                    {
                        Item selectedItem = itemList[playerCharacter.GetItemWithName(NPCShopItemsListBox.SelectedValue.ToString())];
                        //check if player can afford the value * npcPriceIncrease
                        int costOfSelectedItem = Convert.ToInt32(((float)selectedItem.GetItemValue() * currentMap.GetNPCEventAtLocation(currentPos).GetShopPriceIncrease()));

                        if (playerCharacter.GetCurrentCurrencyAmount() >= costOfSelectedItem)
                        {
                            //add item to inventory
                            playerCharacter.AddItem(playerCharacter.GetItemWithName(selectedItem.GetItemName()), 1);
                            //subtract amount from player currency
                            playerCharacter.RemoveCurrency(costOfSelectedItem);
                            //purchase message
                            NPCShopItemDescription.Text = "Purchased " + selectedItem.GetItemName() + ".";
                            //save after each purchase
                            SavePlayerCharacter();
                        }
                        else
                        {
                            //error message
                            NPCShopItemDescription.Text = "You can't afford to purchase " + selectedItem.GetItemName() + ".";
                        }
                    }
                }
                else if (NPCShopTabControl.SelectedItem == NPCShopEquipmentTab)
                {
                    //only attempt to Buy if an Equipment is selected.
                    if (NPCShopEquipmentListBox.SelectedIndex >= 0)
                    {
                        Equipment selectedEquipment = equipmentList[playerCharacter.GetEquipmentWithName(NPCShopEquipmentListBox.SelectedValue.ToString())];
                        //check if player can afford the value * npcPriceIncrease
                        int costOfSelectedEquipment = Convert.ToInt32(((float)selectedEquipment.GetItemValue() * currentMap.GetNPCEventAtLocation(currentPos).GetShopPriceIncrease()));

                        if (playerCharacter.GetCurrentCurrencyAmount() >= costOfSelectedEquipment)
                        {
                            //add equipment to inventory
                            playerCharacter.AddEquipment(playerCharacter.GetEquipmentWithName(selectedEquipment.GetItemName()), 1);
                            //subtract amount from player currency
                            playerCharacter.RemoveCurrency(costOfSelectedEquipment);
                            //purchase message
                            NPCShopItemDescription.Text = "Purchased " + selectedEquipment.GetItemName() + ".";
                            //save after each purchase
                            SavePlayerCharacter();
                        }
                        else
                        {
                            //error message
                            NPCShopItemDescription.Text = "You can't afford to purchase " + selectedEquipment.GetItemName() + ".";
                        }
                    }
                }
            }
            //if player is Selling items to the store
            else if (NPCShopBuySellButton.Content.ToString() == "Sell")
            {
                if (NPCShopTabControl.SelectedItem == NPCShopItemTab)
                {
                    Item selectedItem = itemList[playerCharacter.GetItemWithName(NPCShopItemsListBox.SelectedValue.ToString())];
                    int valueOfSelectedItem = selectedItem.GetItemValue();

                    //check quantity of item
                    if (playerCharacter.GetItemQuantity(playerCharacter.GetItemWithName(selectedItem.GetItemName())) > 0)
                    {
                        //remove item from inventory
                        playerCharacter.RemoveItem(playerCharacter.GetItemWithName(selectedItem.GetItemName()));
                        //add value to currency
                        playerCharacter.AddCurrency(valueOfSelectedItem);
                        //sell message
                        NPCShopItemDescription.Text = "You sold " + selectedItem.GetItemName() + " for " + valueOfSelectedItem + " coins.";
                        //if the last of the player's item is sold, remove the item from NPCShopItemsListBox
                        if (playerCharacter.GetItemQuantity(playerCharacter.GetItemWithName(selectedItem.GetItemName())) < 1)
                        {
                            NPCShopItemsListBox.Items.Remove(selectedItem.GetItemName());
                            NPCShopItemsListBox.SelectedIndex = -1;
                        }
                        //save after each sale
                        SavePlayerCharacter();
                    }
                    else
                    {
                        NPCShopItemDescription.Text = "You don't have that.";
                    }
                }
                else if (NPCShopTabControl.SelectedItem == NPCShopEquipmentTab)
                {
                    Equipment selectedEquipment = equipmentList[playerCharacter.GetEquipmentWithName(NPCShopEquipmentListBox.SelectedValue.ToString())];
                    int valueOfSelectedEquipment = selectedEquipment.GetItemValue();

                    //check quantity of equipment
                    if (playerCharacter.GetEquipmentQuantity(playerCharacter.GetEquipmentWithName(selectedEquipment.GetItemName())) > 0)
                    {
                        //remove item from inventory
                        playerCharacter.RemoveEquipment(playerCharacter.GetEquipmentWithName(selectedEquipment.GetItemName()));
                        //add value to currency
                        playerCharacter.AddCurrency(valueOfSelectedEquipment);
                        //sell message
                        NPCShopItemDescription.Text = "You sold " + selectedEquipment.GetItemName() + " for " + valueOfSelectedEquipment + " coins.";
                        //if the last of the player's equipment is sold, remove the item from NPCShopItemsListBox
                        if (playerCharacter.GetEquipmentQuantity(playerCharacter.GetEquipmentWithName(selectedEquipment.GetItemName())) < 1)
                        {
                            //if removing last of equipment that is equipped, unequip it
                            if (playerCharacter.IsEquipped(selectedEquipment))
                            {
                                playerCharacter.Unequip(selectedEquipment);
                            }
                            NPCShopEquipmentListBox.Items.Remove(selectedEquipment.GetItemName());
                            NPCShopEquipmentListBox.SelectedIndex = -1;
                        }
                        //save after each sale
                        SavePlayerCharacter();
                    }
                    else
                    {
                        NPCShopItemDescription.Text = "You don't have that.";
                    }
                }
                //update view, including
                UpdateCharacterGrid();
            }

            //update view
            UpdateCharacterGrid();
        }

    }
}
